// content.js

const FIELD_SELECTORS = {
  description: idx => `textarea[name="variable_description[${idx}]"]`,
  gla_condition: idx => `select[name="gla_variation_attributes[${idx}][condition]"]`,
  gla_gender: idx => `select[name="gla_variation_attributes[${idx}][gender]"]`,
  gla_size: idx => `select[name="gla_variation_attributes[${idx}][size]"]`,
  gla_sizeSystem: idx => `select[name="gla_variation_attributes[${idx}][sizeSystem]"]`,
  gla_sizeType: idx => `select[name="gla_variation_attributes[${idx}][sizeType]"]`,
  gla_color: idx => `select[name="gla_variation_attributes[${idx}][color]"]`,
  gla_material: idx => `input[name="gla_variation_attributes[${idx}][material]"]`,
  gla_pattern: idx => `input[name="gla_variation_attributes[${idx}][pattern]"]`,
  gla_ageGroup: idx => `select[name="gla_variation_attributes[${idx}][ageGroup]"]`,
  gla_multipack: idx => `input[name="gla_variation_attributes[${idx}][multipack]"]`,
  gla_isBundle: idx => `select[name="gla_variation_attributes[${idx}][isBundle]"]`,
  gla_adult: idx => `select[name="gla_variation_attributes[${idx}][adult]"]`,
  pinterest_condition: idx => `select[name="pinterest_variation_attributes[${idx}][condition]"]`
};

// Expand all variation panels (and open the Variations tab once if needed)
const VAR_HASH = '#variable_product_options';

// Helper to click all expand buttons
function doExpand() {
  const buttons = document.querySelectorAll('.expand_all');
  if (!buttons.length) {
    console.warn('No .expand_all buttons found');
  }
  buttons.forEach(btn => {
    try { btn.click(); }
    catch (e) { console.error('Error clicking expand_all', e); }
  });
}


// Full bulk update
async function runBulkUpdate() {
  const {
    selectedFields = [],
    fieldTemplates = {},
    sidedness = '',
    sidednessAreas = {}
  } = await chrome.storage.local.get([
    'selectedFields',
    'fieldTemplates',
    'sidedness',
    'sidednessAreas'
  ]);

  document.querySelectorAll('.woocommerce_variation').forEach((panel, idx) => {
    const colorEl = panel.querySelector(`select[name="attribute_colors[${idx}]"]`);
    const sizeEl  = panel.querySelector(`select[name="attribute_sizes[${idx}]"]`);
    const color   = colorEl ? colorEl.value : '';
    const size    = sizeEl ? sizeEl.value : '';

    selectedFields.forEach(fieldKey => {
      const selector = FIELD_SELECTORS[fieldKey](idx);
      const el = document.querySelector(selector);
      if (!el) return;

      if (fieldKey === 'description') {
        const segments = fieldTemplates.description || [];
        const parts = segments.map(seg => {
          if (seg.type === 'static') {
            return seg.value;
          } else {
            switch (seg.value) {
              case 'brand_text':
                return fieldTemplates.brand_text || '';
              case 'garment_type':
                return fieldTemplates.garment_type || '';
              case 'garment_subtype':
                return fieldTemplates.garment_subtype || '';
              case 'sidedness':
                return fieldTemplates.sidedness || '';
              case 'garment_text':
                return fieldTemplates.garment_text || '';
              case 'color':
                const cEl = panel.querySelector(`select[name="attribute_colors[${idx}]"]`);
                return cEl ? cEl.value : '';
              case 'size':
                const sEl = panel.querySelector(`select[name="attribute_sizes[${idx}]"]`);
                return sEl ? sEl.value : '';
              case 'sidednessAreas':
                return Object.entries(sidednessAreas)
                  .filter(([code, info]) => code !== 'None' && info.selected)
                  .map(([_, info]) => info.label || '')
                  .join(', ');
              default:
                return '';
            }
          }
        });
        el.value = parts.filter(Boolean).join(' ');
        el.dispatchEvent(new Event('change', { bubbles: true }));
        return;
      }

      // Standard token replacement
      let tpl = fieldTemplates[fieldKey] || '';
      tpl = tpl.replace(/\{color\}/g, color)
               .replace(/\{size\}/g, size)
               .replace(/\{garment_type\}/g, fieldTemplates.garment_type || '')
               .replace(/\{garment_subtype\}/g, fieldTemplates.garment_subtype || '')
               .replace(/\{garment_text\}/g, fieldTemplates.garment_text || '');
      el.value = tpl;
      el.dispatchEvent(new Event('change', { bubbles: true }));
    });

    // ✅ Optional enhancement: Update GfW <input> fields only if selected
    if (selectedFields.includes('gla_color')) {
      const glaColorInput = panel.querySelector(`input[name="gla_variation_attributes[${idx}][color]"]`);
      if (glaColorInput && color) {
        glaColorInput.value = color;
        glaColorInput.dispatchEvent(new Event('input', { bubbles: true }));
        glaColorInput.dispatchEvent(new Event('change', { bubbles: true }));
      }
    }

    if (selectedFields.includes('gla_size')) {
      const glaSizeInput = panel.querySelector(`input[name="gla_variation_attributes[${idx}][size]"]`);
      if (glaSizeInput && size) {
        glaSizeInput.value = size;
        glaSizeInput.dispatchEvent(new Event('input', { bubbles: true }));
        glaSizeInput.dispatchEvent(new Event('change', { bubbles: true }));
      }
    }
  });

  // Save
  const saveBtn = document.querySelector('.save_variations');
  if (saveBtn) saveBtn.click();
}

// Chrome message listeners
chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
//  if (msg.action === 'expand_variations') {
//    expandAllVariationPanels();
//  }
  if (msg.action === 'toggle_variations') {
    const isActive = window.location.hash === VAR_HASH;
    const anchor   = document.querySelector(`a[href="${VAR_HASH}"]`);

    if (!isActive) {
      // 1) Click to open
      if (anchor) anchor.click();
      else console.warn('Variations tab anchor not found');

      // 2) After a short delay (panel needs to render), expand all
      setTimeout(doExpand, 500);

      // Now it will be active
      sendResponse({ active: true });
    } else {
      // Already open → just expand immediately
      doExpand();
      sendResponse({ active: true });
    }
    return true; // indicates we’ll call sendResponse synchronously
  }

  if (msg.action === 'check_variations_active') {
    sendResponse({ active: window.location.hash === VAR_HASH });
    return true;
  }

  if (msg.action === 'apply_field') {
    applyFieldToAllVariations(msg.fieldKey, msg.newValue);
  }
  if (msg.action === 'run_bulk_update') {
    runBulkUpdate();
  }

  return true;
});