// background.js
import { DEFAULT_EFFECT_SETTINGS, DEFAULT_CONSTRAINTS } from './licencing/constants.js';
import { getHashDeviceSigning } from './libs/device_utils.js';
import { checkAndCacheLicence } from './licencing/licence-check.js';

const STORAGE_KEYS = {
  browserId:           'browserId',
  licenceKey:          'licenceKey',
  licenceType:         'licenceType',
  constraints:         'licenceConstraints',
  lastChecked:         'licenceLastChecked',
  deviceId:            'deviceId',
  deviceFingerprint:   'deviceFingerprint',
  fingerprint:         'fingerprint',
  os:                  'os',
  browserInfo:         'browserInfo',
  timezoneLocale:      'timezoneLocale'
};


// Helper to do the full Lite reset:
async function resetToLite() {
  // 1) Core keys to set or remove
  await chrome.storage.local.set({
    activeProfile: 'default',
    blockoutEffect: 'blur',
    licenceConstraints: DEFAULT_CONSTRAINTS,
    profiles: ['default']
  });

  await chrome.storage.local.remove([
    'email',
    'licenceKey',
    'licenceLastChecked'
  ]);

  // 2) Remove any srb_regions_{profile}_ keys that are *not* default
  const all = await chrome.storage.local.get(null);
  const nonDefaultKeys = Object.keys(all).filter(key =>
    key.startsWith('srb_regions_') &&
    !key.startsWith('srb_regions_default_')
  );
  if (nonDefaultKeys.length) {
    await chrome.storage.local.remove(nonDefaultKeys);
  }

  // 3) For the default profile’s saved regions, overwrite each region’s effect
  const defaultRegionKeys = Object.keys(all).filter(key =>
    key.startsWith('srb_regions_default_')
  );
  for (const key of defaultRegionKeys) {
    const { [key]: regions = [] } = all;
    const updated = regions.map(r => ({
      top:    r.top,
      left:   r.left,
      width:  r.width,
      height: r.height,
      effect: { ...DEFAULT_EFFECT_SETTINGS }
    }));
    await chrome.storage.local.set({ [key]: updated });
  }
}

async function ensureFingerprintMetadata() {
  try {
    const {
  deviceId,
  deviceFingerprint,
  os,
  timezoneLocale,
  browserInfo
} = await getHashDeviceSigning();

    //const browserInfo = navigator.userAgent;

    // Sanity check logging (optional)
    if (!deviceId || !deviceFingerprint) {
      console.warn('[SRB] Warning: Missing fingerprint data');
    }

    await chrome.storage.local.set({
      [STORAGE_KEYS.deviceId]: deviceId,
      [STORAGE_KEYS.deviceFingerprint]: deviceFingerprint,
      [STORAGE_KEYS.browserId]: deviceId, // still alias
      [STORAGE_KEYS.os]: os,
      [STORAGE_KEYS.browserInfo]: browserInfo,
      [STORAGE_KEYS.timezoneLocale]: timezoneLocale,
      [STORAGE_KEYS.fingerprint]: {
        deviceId,
        deviceFingerprint,
        os,
        timezoneLocale,
        browserInfo
      }
    });

    console.log('[SRB] Fingerprint metadata stored.');
  } catch (err) {
    console.error('[SRB] Failed to generate fingerprint metadata:', err);
  }
}

async function performLicenceCheck() {
  try {
    const { licenceType, constraints } = await checkAndCacheLicence();
    console.log('[SRB] Licence check result:', licenceType, constraints);
  } catch (err) {
    console.error('[SRB] Licence check failed:', err);
  }
}

async function initialiseProfiles() {
  const { profiles, activeProfile } = await chrome.storage.local.get(['profiles', 'activeProfile']);

  if (!Array.isArray(profiles) || !profiles.includes('default')) {
    await chrome.storage.local.set({ profiles: ['default'] });
  }

  if (!activeProfile) {
    await chrome.storage.local.set({ activeProfile: 'default' });
  }
}

// Initialisation runner
async function init() {
  await ensureFingerprintMetadata();
  await performLicenceCheck();
  // If we are Lite, reset everything
  const { licenceType } = await chrome.storage.local.get('licenceType');
  if (licenceType === 'Lite') {
    console.log('[SRB] Lite detected on install → wiping extras');
    await resetToLite();
  }
  await initialiseProfiles();
}

chrome.runtime.onInstalled.addListener(async () => {
  console.log('[SRB] Extension installed or updated.');
  await ensureFingerprintMetadata();
  await performLicenceCheck();
  // If we are Lite, reset everything
  const { licenceType } = await chrome.storage.local.get('licenceType');
  if (licenceType === 'Lite') {
    console.log('[SRB] Lite detected on install → wiping extras');
    await resetToLite();
  }

});

chrome.runtime.onStartup.addListener(async () => {
  console.log('[SRB] Extension starting up.');
  await ensureFingerprintMetadata();
  await performLicenceCheck();
  // If we are Lite, reset everything
  const { licenceType } = await chrome.storage.local.get('licenceType');
  if (licenceType === 'Lite') {
    console.log('[SRB] Lite detected on install → wiping extras');
    await resetToLite();
  }
});

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'getTabUrl') {
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
      if (tabs.length > 0) {
        sendResponse({ url: tabs[0].url });
      } else {
        sendResponse({ url: null });
      }
    });
    return true;
  }
});

// Screenshot proxy
chrome.runtime.onMessage.addListener(async (message, sender, sendResponse) => {
  if (message.action === 'captureScreenshot') {
    try {
      const dataUrl = await chrome.tabs.captureVisibleTab(null, { format: 'png' });
      console.log('[Background] Screenshot captured.');
      sendResponse({ success: true, dataUrl });
    } catch (e) {
      console.error('[Background] Screenshot failed:', e);
      sendResponse({ success: false });
    }
  }
  return true; // needed for async sendResponse
});