// ==== CONFIGURATION ====
const API_BASE = 'https://www.jaysndees.com/extensions/srb_v2043/api';

// ==== DEFAULT CONSTRAINTS (for “Lite” fallback) ====
const DEFAULT_CONSTRAINTS = {
  licence_type: 'Lite',
  max_regions_per_page: 3,
  max_pages: 2,
  max_profiles: 1,
  allowed_profiles: false,
  allowed_effects: ['blur'],
  blur_slider: false,
  import_export: false,
  activation_limit: 1,
  notes: ''
};

// ==== UTILITY ====
function showToast(text) {
  const t = document.createElement('div');
  t.className = 'srb-toast';
  t.textContent = text;
  document.body.appendChild(t);
  requestAnimationFrame(() => t.style.opacity = '1');
  setTimeout(() => {
    t.style.opacity = '0';
    t.addEventListener('transitionend', () => t.remove());
  }, 3000);
}

// Fetch every setting—base + advanced—in one shot
async function loadAllEffectSettings() {
  // 1) define every setting Options page supports, with inline defaults
  const defaults = {
    blockoutEffect:   'blur',
    blurRadius:       8,
    solidOpacity:     1,
    solidColour:      '#000000',
    pixelSize:        6,
    // grid defaults
    gridLineColour:   '#444444',
    gridBgColour:     '#000000',
    gridBgOpacity:    0.8,
    //gridSpacing:      6,
    // checkerboard defaults
    checkerColour1:   '#000000',
    checkerColour2:   '#ffffff',
    checkerSpacing:   20,
    checkerOpacity:   0.8,
    // truechecker defaults
    truecheckerColour1: '#000000',
    truecheckerColour2: '#444444',
    // moiré defaults
    moireStripeColour:  '#000000',
    moireBgOpacity:     0.8,
    //moireStripeWidth:   8,
    //moireGap:           8,
    // dots defaults
    dotsBgColour:       '#000000',
    dotsDotColour:      '#ffffff',
    dotsSpacing:        20,
  };

  // 2) pull whatever the user has actually set
  const stored = await chrome.storage.local.get(Object.keys(defaults));

  // 3) merge
  return { ...defaults, ...stored };
}

async function getEffectSettings() {
  const defaults = {
    blockoutEffect: 'blur',
    blurRadius: 8,
    solidOpacity: 1,
    solidColour: '#000000',
    pixelSize: 6
  };
  const data = await chrome.storage.local.get(Object.keys(defaults));
  return { ...defaults, ...data };
}

async function buildRegionKey() {
  const url = location.origin + location.pathname;
  const { activeProfile = 'default' } = await chrome.storage.local.get('activeProfile');
  return `srb_regions_${activeProfile}_${url}`;
}

// --------------------------------------------------------------------------------------------------------------------------
// ── Enhanced Pattern Generators ────────────────────────────────────────────

// (1) Grid effect — 1px lines over block
function getGridData(
  spacing = 6,
  lineColour = '#444444',
  bgColour   = '#000000',
  bgOpacity  = 0.8
) {
  const alphaHex = Math.round(bgOpacity * 255).toString(16).padStart(2,'0');
  const hex = bgColour.replace('#','');
  return {
    backgroundColor: `#${hex}${alphaHex}`,
    backgroundImage: `
      linear-gradient(to bottom, ${lineColour} 1px, transparent 1px),
      linear-gradient(to right, ${lineColour} 1px, transparent 1px)
    `,
    backgroundSize: `${spacing}px ${spacing}px, ${spacing}px ${spacing}px`,
    backgroundRepeat: 'repeat',
    pointerEvents: 'none'
  };
}

// (2) Classic Carbon effect — repeating-conic-pixelation
function getCarbonDataOriginal(pixelSize = 6) {
  return {
    backgroundImage: `repeating-conic-gradient(
      #000 0% 25%, #444 0% 50%
    )`,
    backgroundSize:  `${pixelSize}px ${pixelSize}px`,
    backgroundRepeat:'repeat',
    imageRendering:  'pixelated',
    filter:          'none',
    backgroundColor: 'transparent'
  };
}

// (3) Checkerboard effect — linear with transparent areas
function getCheckerboardData(size = 20, col1 = '#000000', col2 = '#ffffff') {
  return {
    backgroundImage: `
      linear-gradient(45deg, ${col1} 25%, transparent 25%),
      linear-gradient(-45deg, ${col1} 25%, transparent 25%),
      linear-gradient(45deg, transparent 75%, ${col2} 75%),
      linear-gradient(-45deg, transparent 75%, ${col2} 75%)
    `,
    backgroundSize:   `${size}px ${size}px`,
    backgroundPosition: `0 0, 0 ${size/2}px, ${size/2}px -${size/2}px, -${size/2}px 0`,
    backgroundRepeat: 'repeat',
    filter:           'none',
    pointerEvents:    'none'
  };
}

// (4) True Checkerboard effect — repeating-conic-pixelation colour 1,colour 2 pattern
function getTrueCheckerboardData(size = 6, col1 = '#000000', col2 = '#444444') {
  return {
    backgroundImage: `repeating-conic-gradient(
      ${col1} 0% 25%,
      ${col2} 25% 50%,
      ${col1} 50% 75%,
      ${col2} 75% 100%
    )`,
    backgroundSize:   `${size}px ${size}px`,
    backgroundRepeat: 'repeat',
    imageRendering:   'pixelated',
    filter:           'none',
    backgroundColor:  'transparent'
  };
}

// (5) Moiré — dark overlay with angled transparent stripes
function getMoireData(stripeWidth = 8, gap = 8, stripeColor = '#000', bgOpacity = 0.8) {
  // We build two crossing stripe patterns at 45° and -45°
  //const stripeColor = 'rgba(0,0,0,0.8)';
const bg = stripeColor.replace(/^#/, '');
  return {
    //backgroundColor: stripeColor,
    backgroundColor: `#000000${Math.round(bgOpacity*255).toString(16).padStart(2,'0')}`,
    backgroundImage: `
      repeating-linear-gradient(
        45deg,
        transparent 0,
        transparent ${gap}px,
        ${stripeColor} ${gap}px,
        ${stripeColor} ${gap + stripeWidth}px
      ),
      repeating-linear-gradient(
        -45deg,
        transparent 0,
        transparent ${gap}px,
        ${stripeColor} ${gap}px,
        ${stripeColor} ${gap + stripeWidth}px
      )
    `,
    backgroundSize:  `${gap + stripeWidth}px ${gap + stripeWidth}px`,
    backgroundRepeat:'repeat',
    filter:          'none'
  };
}

// (6) Dots — dark overlay punctured by transparent circles
/**
 * spacing: distance between dot centers
 * bgColour: CSS colour for the background overlay
 * dotColour: CSS colour for the dots
 */
function getDotsData(spacing = 20, bgColour = '#000000', dotColour = '#ffffff', dotSize = spacing / 2) {
  //const dotSize = spacing / 2;   // half of spacing
  const offset  = (spacing - dotSize) / 2;

  return {
    backgroundColor: bgColour,
    backgroundImage: `
      radial-gradient(
        circle ${dotSize}px at ${offset}px ${offset}px,
        ${dotColour} ${dotSize}px,
        transparent ${dotSize}px
      )
    `,
    backgroundSize:   `${spacing}px ${spacing}px`,
    backgroundRepeat: 'repeat',
    filter:           'none'
  };
}
// ────────────────────────────────────────────
// Apply CSS styles to a region DIV based on settings
function applyEffectStyle(el, settings) {
  const { blockoutEffect, blurRadius, solidOpacity, solidColour, pixelSize,
          gridLineColour, gridBgColour, gridBgOpacity, checkerColour1,
          checkerColour2, checkerSpacing, checkerOpacity,
          truecheckerColour1, truecheckerColour2,
          moireStripeColour, moireBgOpacity, dotsBgColour, dotsDotColour, dotsSpacing
        } = settings;

  Object.assign(el.style, {
    position:      'absolute',
    zIndex:        '99999',
    pointerEvents: 'none'
  });

  switch (blockoutEffect) {
    case 'solid':
      el.style.backdropFilter  = 'none';
      el.style.backgroundColor = solidColour;
      el.style.opacity         = solidOpacity;
      break;

    case 'grid':
      Object.assign(el.style, getGridData(
        pixelSize, gridLineColour, gridBgColour, gridBgOpacity
      ));
      break;

    case 'carbon':
      Object.assign(el.style, getCarbonDataOriginal(pixelSize));
      break;

    case 'checkerboard':
      Object.assign(el.style, getCheckerboardData(
        checkerSpacing, checkerColour1, checkerColour2
      ));
      el.style.opacity = checkerOpacity;
      break;

    case 'truechecker':
      Object.assign(el.style, getTrueCheckerboardData(
        pixelSize, truecheckerColour1, truecheckerColour2
      ));
      break;

   case 'moire':
      Object.assign(el.style, getMoireData(
        pixelSize, pixelSize,
        moireStripeColour, moireBgOpacity
      ));
      break;

    case 'dots':
      //const spacing = pixelSize;
      Object.assign(el.style, getDotsData(
        dotsSpacing, dotsBgColour, dotsDotColour, pixelSize
      ));
      break;

    default:  // blur
      el.style.backdropFilter  = `blur(${blurRadius}px)`;
      el.style.backgroundColor = 'rgba(0,0,0,0.4)';
  }
}

// ==== STATE ====
let constraints = null;
let effectSettings = null;
let isArmedToDraw = false;
let isDrawing = false;
let startX, startY, currentRegion;

// ==== INITIALISE ====
(async () => {
  const { licenceConstraints } = await chrome.storage.local.get('licenceConstraints');
  constraints = licenceConstraints || DEFAULT_CONSTRAINTS;

  effectSettings = await loadAllEffectSettings();

  const allSettings = await loadAllEffectSettings();
  const key = await buildRegionKey();
  const { [key]: regions = [] } = await chrome.storage.local.get(key);

  regions.forEach(r => {
    const el = document.createElement('div');
    applyEffectStyle(el, { ...allSettings, ...(r.effect||{}) });
    Object.assign(el.style, {
      top:    r.top,
      left:   r.left,
      width:  r.width,
      height: r.height
    });
    el.classList.add('srb_region');
    document.body.appendChild(el);
  });
})();

// ==== MESSAGES ====
chrome.runtime.onMessage.addListener((req, sender, sendResponse) => {
  (async () => {
    const key = await buildRegionKey();
    const all = await chrome.storage.local.get(null);

    switch (req.action) {

      case 'updateConstraints': {
    // re‑load the cached constraints
    chrome.storage.local.get('licenceConstraints').then(({ licenceConstraints }) => {
      constraints = licenceConstraints;
    });
    return;  // no sendResponse needed
      }

      case 'getRegionCount': {
        const regions = all[key] || [];
        sendResponse({ success: true, count: regions.length });
        break;
      }

      case 'addRegion': {
  // 🔄 Refresh the constraints on every addRegion
//  constraints = await fetchConstraints(
//     (await chrome.storage.sync.get('licenceType')).licenceType || 'Lite'
//  );
  // 🔄 Refresh the constraints from the stored licenceConstraints
  const { licenceConstraints } = await chrome.storage.local.get('licenceConstraints');
  constraints = licenceConstraints || constraints;
  if (!constraints) {
    sendResponse({ success: false, error: 'constraints_unavailable' });
    return;
  }

  const regions = all[key] || [];
  if (constraints.max_regions_per_page >= 0 &&
      regions.length >= constraints.max_regions_per_page) {
    showToast('Max regions reached for your licence');
    sendResponse({ success: false, error: 'max_regions' });
    return;
  }

        const prefix = key.slice(0, key.lastIndexOf('_') + 1);
        const pageKeys = Object.keys(all).filter(k => k.startsWith(prefix));
        const uniquePages = new Set(pageKeys);
        if (!pageKeys.includes(key) && uniquePages.size >= constraints.max_pages) {
          showToast('Max pages reached for your licence');
          sendResponse({ success: false, error: 'max_pages' });
          return;
        }

        isArmedToDraw = true;
        document.body.style.cursor = 'crosshair';
        showToast('Click & drag to draw a region');
        sendResponse({ success: true });
        break;
      }

      case 'clearRegions': {
        await chrome.storage.local.remove(key);
        document.querySelectorAll('.srb_region').forEach(e => e.remove());
        sendResponse({ success: true });
        showToast('Regions cleared on this page');
        break;
      }

    case 'refreshRegions': {
      const key = await buildRegionKey();
      const { [key]: regions = [] } = await chrome.storage.local.get(key);
      document.querySelectorAll('.srb_region').forEach(e => e.remove());
const allSettings = await loadAllEffectSettings();
regions.forEach(r => {
  const el = document.createElement('div');
  applyEffectStyle(el, { ...allSettings, ...(r.effect||{}) });
  Object.assign(el.style, {
    top:    r.top, left: r.left,
    width:  r.width, height: r.height
  });
  el.classList.add('srb_region');
  document.body.appendChild(el);
});
      sendResponse({ success: true });
      break;
    }

    case 'clearAllRegions': {
      const { activeProfile = 'default' } = await chrome.storage.local.get('activeProfile');
      const prefix = `srb_regions_${activeProfile}_`;
      const all    = await chrome.storage.local.get(null);
      const keysToRemove = Object.keys(all).filter(k => k.startsWith(prefix));
      chrome.storage.local.remove(keysToRemove, () => {
        document.querySelectorAll('.srb_region').forEach(e => e.remove());
        sendResponse({ success: true });
        showToast('All regions cleared for this profile');
      });
      break;
    }

    case 'clearAllProfiles': {
      const all    = await chrome.storage.local.get(null);
      const regionKeys = Object.keys(all).filter(k => k.startsWith('srb_regions_'));
      await chrome.storage.local.remove(regionKeys);
      await chrome.storage.local.set({ profiles: ['default'], activeProfile: 'default' });
      document.querySelectorAll('.srb_region').forEach(e => e.remove());
      sendResponse({ success: true });
      showToast('All profiles and regions reset');
      break;
    }

      case 'toast': {
        if (req.message) showToast(req.message);
        sendResponse({ success: true });
        break;
      }

      default: {
        sendResponse({ success: false, error: 'unknown_action' });
      }
    }
  })();
  return true;
});

// ==== DRAWING ====
document.addEventListener('pointerdown', async (e) => {
  if (!isArmedToDraw) return;
  isArmedToDraw = false;
  document.body.style.cursor = '';
  isDrawing = true;
  startX = e.pageX;
  startY = e.pageY;

  // grab the very latest full settings (including advanced colours, sizes, etc)
  effectSettings = await loadAllEffectSettings();

  // create the region and apply the user’s chosen effect
  currentRegion = document.createElement('div');
  currentRegion.classList.add('srb_region');
  await applyEffectStyle(currentRegion, effectSettings);
  document.body.appendChild(currentRegion);

  document.body.style.userSelect = 'none';
  document.addEventListener('pointermove', drawRegion);
  document.addEventListener('pointerup', stopDrawing, { once: true });
});

function drawRegion(e) {
  if (!isDrawing) return;
  const x = Math.min(e.pageX, startX);
  const y = Math.min(e.pageY, startY);
  const w = Math.abs(e.pageX - startX);
  const h = Math.abs(e.pageY - startY);
  Object.assign(currentRegion.style, {
    left: x + 'px',
    top: y + 'px',
    width: w + 'px',
    height: h + 'px'
  });
}

async function stopDrawing() {
  isDrawing = false;
  document.body.style.userSelect = '';
  document.removeEventListener('pointermove', drawRegion);

  const fullSettings = await loadAllEffectSettings();

  const regionData = {
    top: currentRegion.style.top,
    left: currentRegion.style.left,
    width: currentRegion.style.width,
    height: currentRegion.style.height,
    effect: { ...fullSettings }
  };
  const key = await buildRegionKey();
  const stored = await chrome.storage.local.get(key);
  const arr = stored[key] || [];
  arr.push(regionData);
  await chrome.storage.local.set({ [key]: arr });
  showToast('Region added');
}
