import { getHashDeviceSigning } from '../libs/device_utils.js';

const EXT_API_KEY = 'a3fbe91c321b66d2188395f36288d7a3f5e36288d7a3f5e1cc4ffb0e2412159f8ceabf1bdfe8';
const API_BASE    = 'https://www.jaysndees.com/extensions/srb_v2043/api';

const DEFAULT_CONSTRAINTS = {
  licence_type: 'Lite',
  max_regions_per_page: 3,
  max_pages: 2,
  allowed_profiles: false,
  max_profiles: 1,
  allowed_effects: ['blur'],
  blur_slider: false,
  import_export: false,
  activation_limit: 1,
  screenshot: false,
  watermark: true,
  watermark_slider: false,
  watermark_toggle: false,
  watermark_opacity_range: [1.0],
  color_picker: false,
  pixel_able: false,
  pixel_size: 0
};

const STORAGE_KEYS = {
  browserId:        'browserId',
  licenceKey:       'licenceKey',
  licenceType:      'licenceType',
  constraints:      'licenceConstraints',
  lastChecked:      'licenceLastChecked',
  deviceId:         'deviceId',
  fingerprint:      'fingerprint',
  os:               'os',
  browserInfo:      'browserInfo',
  timezoneLocale:   'timezoneLocale',

};

async function apiFetch(path, opts = {}) {
  const url = `${API_BASE}/${path}`;
  opts.headers = {
    'Content-Type': 'application/json',
    'X-Extension-Key': EXT_API_KEY,
    ...(opts.headers || {})
  };
  const resp = await fetch(url, opts);
  const json = await resp.json();
  if (!resp.ok) {
    if (json.error === 'expired') {
    // handle expiry specially
      throw new Error('expired:' + json.expires_at);
    }
    throw new Error(`API error: ${resp.status}`);
  }
  return json;

}

export async function activateLicence(licenceKey, email = '') {
  const stored = await chrome.storage.local.get([
    STORAGE_KEYS.browserId,
    STORAGE_KEYS.deviceId,
    STORAGE_KEYS.fingerprint,
    STORAGE_KEYS.os,
    STORAGE_KEYS.browserInfo,
    STORAGE_KEYS.timezoneLocale
  ]);

  const browserId       = stored[STORAGE_KEYS.browserId];
  const deviceId        = stored[STORAGE_KEYS.deviceId];
  const fingerprint     = stored[STORAGE_KEYS.fingerprint];
  const os              = stored[STORAGE_KEYS.os];
  const browserInfo     = stored[STORAGE_KEYS.browserInfo] || navigator.userAgent;
  const timezoneLocale  = stored[STORAGE_KEYS.timezoneLocale] || Intl.DateTimeFormat().resolvedOptions().timeZone;

  if (!browserId || !deviceId || !fingerprint?.deviceFingerprint) {
    throw new Error('[Activation] Missing fingerprinting info. Ensure metadata was initialised.');
  }

  const payload = {
    licence_key:        licenceKey,
    email,
    browser_id:         browserId,
    device_id:          deviceId,
    device_fingerprint: fingerprint.deviceFingerprint,
    os,
    browser_info:       browserInfo,
    timezone_locale:    timezoneLocale
  };

  const response = await apiFetch('activate_licence.php', {
    method: 'POST',
    body: JSON.stringify(payload)
  });

  if (response.success) {
    await chrome.storage.local.set({
      [STORAGE_KEYS.licenceKey]:  licenceKey,
      [STORAGE_KEYS.licenceType]: response.licence_type
    });
  }

  return response;
}

export async function validateLicence(licenceKey) {
  const { deviceId, browserId } = await chrome.storage.local.get(['deviceId','browserId']);
  if (!licenceKey || !browserId || !deviceId) {
    throw new Error('Missing licenceKey or fingerprint data');
  }

  const qs = new URLSearchParams({
    licence_key: licenceKey,
    browser_id: browserId,
    device_id: deviceId
  });

  return apiFetch(`validate_licence.php?${qs}`);
}

export function getLicenceConstraints(licenceType) {
  const qs = new URLSearchParams({ licence_type: licenceType });
  return apiFetch(`get_licence_constraints.php?${qs}`);
}

export async function checkAndCacheLicence(forceRefresh = false) {
  const now = Date.now();
  const stored = await chrome.storage.local.get([
    STORAGE_KEYS.licenceKey,
    STORAGE_KEYS.browserId,
    STORAGE_KEYS.deviceId,
    STORAGE_KEYS.fingerprint,
    STORAGE_KEYS.licenceType,
    STORAGE_KEYS.constraints,
    STORAGE_KEYS.lastChecked
  ]);

  const {
    licenceKey,
    browserId,
    deviceId,
    fingerprint,
    licenceType,
    licenceConstraints,
    licenceLastChecked
  } = {
    licenceKey:         stored[STORAGE_KEYS.licenceKey],
    browserId:          stored[STORAGE_KEYS.browserId],
    deviceId:           stored[STORAGE_KEYS.deviceId],
    fingerprint:        stored[STORAGE_KEYS.fingerprint],
    licenceType:        stored[STORAGE_KEYS.licenceType],
    licenceConstraints: stored[STORAGE_KEYS.constraints],
    licenceLastChecked: stored[STORAGE_KEYS.lastChecked]
  };

  const isFresh = licenceLastChecked && (now - licenceLastChecked < 24 * 3600 * 1000);

  if (!forceRefresh && licenceType && licenceConstraints && isFresh) {
    return {
      licenceType,
      constraints: licenceConstraints
    };
  }

  if (!licenceKey || !browserId || !deviceId || !fingerprint?.deviceFingerprint) {
    await chrome.storage.local.set({
      [STORAGE_KEYS.licenceType]: DEFAULT_CONSTRAINTS.licence_type,
      [STORAGE_KEYS.constraints]: DEFAULT_CONSTRAINTS,
      [STORAGE_KEYS.lastChecked]: now
    });
    return {
      licenceType: DEFAULT_CONSTRAINTS.licence_type,
      constraints: DEFAULT_CONSTRAINTS
    };
  }

  try {
	const validation = await validateLicence(licenceKey);

    if (!validation.success || validation.status !== 'active') {
      throw new Error('Licence validation failed or inactive');
    }

    const validatedType = validation.licence_type;
    let constraints = DEFAULT_CONSTRAINTS;

    if (validatedType !== DEFAULT_CONSTRAINTS.licence_type) {
      const constraintRes = await getLicenceConstraints(validatedType);

      if (constraintRes.success && constraintRes.constraints) {
        constraints = constraintRes.constraints;
      } else {
        throw new Error('Could not load constraints for ' + validatedType);
      }
    }

//    await chrome.storage.local.set({
 //     [STORAGE_KEYS.licenceType]: validatedType,
//      [STORAGE_KEYS.constraints]: constraints,
//      [STORAGE_KEYS.lastChecked]: now
//    });

  // Store the standard licence and constraints
  const itemsToStore = {
    [STORAGE_KEYS.licenceType]: validatedType,
    [STORAGE_KEYS.constraints]: constraints,
    [STORAGE_KEYS.lastChecked]: now
  };

  // Store pixelate-specific keys (if any)
  if ('pixel_able' in constraints) {
    itemsToStore.pixel_able = constraints.pixel_able;
  }
  if ('pixel_size' in constraints) {
    itemsToStore.pixel_size = constraints.pixel_size;
  }


  await chrome.storage.local.set(itemsToStore);

    return { licenceType: validatedType, constraints };
  } catch (err) {
    if (err.message.startsWith('expired:')) {
      const expiresAt = err.message.split(':')[1];
      alert(`Your licence expired on ${expiresAt}. Reverting to Lite.`);
    }
    console.warn('[Licence] Validation error. Falling back to Lite:', err);
    await chrome.storage.local.set({
      [STORAGE_KEYS.licenceType]: DEFAULT_CONSTRAINTS.licence_type,
      [STORAGE_KEYS.constraints]: DEFAULT_CONSTRAINTS,
      [STORAGE_KEYS.lastChecked]: now
    });
    return {
      licenceType: DEFAULT_CONSTRAINTS.licence_type,
      constraints: DEFAULT_CONSTRAINTS
    };
  }
}