// options.js

chrome.storage.local.get(
  [
    'activeProfile','browserId','browserInfo','deviceFingerprint','deviceId','fingerprint','licenceConstraints','licenceLastChecked','licenceType','os','profiles','timezoneLocale'
  ],
 settings => initialise(settings)
);

const TIER_CONSTRAINTS = {
  Lite: {
    max_regions_per_page: 3,
    max_pages: 2,
    allowed_profiles: false,
    max_profiles: 1,
    allowed_effects: ["blur"],
    blur_slider: false,
    import_export: false,
    activation_limit: 1,
    screenshot: false,
    watermark: true,
    watermark_slider: false,
    watermark_toggle: false,
    watermark_opacity_range: [1.0],
    color_picker: false,
    pixel_able: false,
    pixel_size: 0
  },
  Standard: {
    max_regions_per_page: 8,
    max_pages: 3,
    allowed_profiles: true,
    max_profiles: 3,
    allowed_effects: ["blur", "solid"],
    blur_slider: true,
    import_export: true,
    activation_limit: 1,
    screenshot: true,
    watermark: true,
    watermark_slider: false,
    watermark_toggle: false,
    watermark_opacity_range: [0.8],
    color_picker: false,
    pixel_able: false,
    pixel_size: 0
  },
  Anonymous: {
    max_regions_per_page: 9999,
    max_pages: 30,
    allowed_profiles: true,
    max_profiles: 10,
    allowed_effects: ["blur", "solid", "pixelate", "grid", "checkerboard", "truechecker", "carbon", "dots", "moire"],
    blur_slider: true,
    import_export: true,
    activation_limit: 10,
    screenshot: true,
    watermark: true,
    watermark_slider: true,
    watermark_toggle: true,
    watermark_opacity_range: [0.1, 1.0],
    color_picker: true,
    pixel_able: true,
    pixel_size: 10
  }
};



// ==== DOM References ====
const effectSelect          = document.getElementById('effectSelect');
const blurSettings          = document.getElementById('blurSettings');
const solidSettings         = document.getElementById('solidSettings');
const pixelateSettings      = document.getElementById('pixelateSettings');

const blurRadiusInput       = document.getElementById('blurRadius');
const blurRadiusVal         = document.getElementById('blurRadiusVal');

const solidOpacityInput     = document.getElementById('solidOpacity');
const solidOpacityVal       = document.getElementById('solidOpacityVal');
const solidColourInput      = document.getElementById('solidColour');

const pixelSizeSlider       = document.getElementById('pixelSize');
const pixelSizeVal          = document.getElementById('pixelSizeVal');

const gridSettings        = document.getElementById('gridSettings');
const gridLineColourInput = document.getElementById('gridLineColour');
const gridBgColourInput   = document.getElementById('gridBgColour');
const gridBgOpacityInput  = document.getElementById('gridBgOpacity');
const gridBgOpacityVal    = document.getElementById('gridBgOpacityVal');

const checkerboardSettings  = document.getElementById('checkerboardSettings');
const checkerColour1Input   = document.getElementById('checkerColour1');
const checkerColour2Input   = document.getElementById('checkerColour2');
const checkerSpacingSlider  = document.getElementById('checkerSpacing');
const checkerSpacingVal     = document.getElementById('checkerSpacingVal');
const checkerOpacityInput   = document.getElementById('checkerOpacity');
const checkerOpacityVal     = document.getElementById('checkerOpacityVal');

const truecheckerSettings      = document.getElementById('truecheckerSettings');
const truecheckerColour1Input  = document.getElementById('truecheckerColour1');
const truecheckerColour2Input  = document.getElementById('truecheckerColour2');

const moireSettings        = document.getElementById('moireSettings');
const moireStripeColour    = document.getElementById('moireStripeColour');
const moireBgOpacityInput  = document.getElementById('moireBgOpacity');
const moireBgOpacityVal    = document.getElementById('moireBgOpacityVal');

const dotsSettings         = document.getElementById('dotsSettings');
const dotsBgColourInput    = document.getElementById('dotsBgColour');
const dotsDotColourInput   = document.getElementById('dotsDotColour');
const dotsSpacingSlider    = document.getElementById('dotsSpacing');
const dotsSpacingVal       = document.getElementById('dotsSpacingVal');

const optImpExp              = document.getElementById('optImpExp');
const exportBtn              = document.getElementById('exportBtn');
const importBtn              = document.getElementById('importBtn');
const importInput            = document.getElementById('importInput');
const optGlobalClear         = document.getElementById('optGlobalClear');
const clearCurrentProfileBtn = document.getElementById('clearCurrentProfile');
const clearAllProfilesBtn    = document.getElementById('clearAllProfiles');
const optProfiles            = document.getElementById('optProfiles');
const profileSelect          = document.getElementById('profileSelect');
const newProfileBtn          = document.getElementById('newProfile');
const deleteProfileBtn       = document.getElementById('deleteProfile');
const statusLabel            = document.getElementById('statusLabel');
const watermarkSection       = document.getElementById('watermarkSection');
const watermarkEnabledInput  = document.getElementById('watermarkEnabled');
const watermarkOpacityInput  = document.getElementById('watermarkOpacity');
const watermarkOpacityVal    = document.getElementById('watermarkOpacityVal');

// ==== Helpers ====
function showStatus(msg, isError = false) {
  statusLabel.textContent = msg;
  statusLabel.style.color = isError ? 'red' : 'green';
  setTimeout(() => statusLabel.textContent = '', 3000);
}

function renderSliders(effect) {
  blurSettings.style.display     = effect === 'blur'   ? 'block' : 'none';
  solidSettings.style.display    = effect === 'solid'  ? 'block' : 'none';
  // pixelSize slider covers pixelate, carbon, moiré, dots, etc
  pixelateSettings.style.display = 
      ['pixelate','grid','truechecker','carbon','moire','dots'].includes(effect)
      ? 'block' 
      : 'none';

  // Advanced panels
  gridSettings.style.display = effect === 'grid' ? 'block' : 'none';
  checkerboardSettings.style.display = effect === 'checkerboard' ? 'block' : 'none';
  truecheckerSettings.style.display = effect === 'truechecker' ? 'block' : 'none';
  moireSettings.style.display  = effect === 'moire'  ? 'block' : 'none';
  dotsSettings.style.display   = effect === 'dots'   ? 'block' : 'none';
}

// Build storage key by profile + URL (same as content.js)
async function buildRegionKey(url) {
  const { activeProfile = 'default' } = await chrome.storage.local.get('activeProfile');
  return `srb_regions_${activeProfile}_${url}`;
}

// ==== Load & Save Effect Settings ====
async function loadEffectSettings() {
  const { licenceConstraints = {} } = await chrome.storage.local.get('licenceConstraints');


  const defaults = {
    blockoutEffect: 'blur',
    blurRadius: 8,
    solidOpacity: 1,
    solidColour: '#000000',
    pixelSize: 10,
    // advanced defaults:
    moireStripeColour: '#000000',
    moireBgOpacity: 0.8,
    dotsBgColour: '#000000',
    dotsDotColour: '#ffffff',
    dotsSpacing: 20,
    gridLineColour: '#444444',
    gridBgColour:   '#000000',
    gridBgOpacity:  0.8,
    checkerColour1: '#000000',
    checkerColour2: '#ffffff',
    checkerSpacing: 20,
    checkerOpacity: 0.8,
    truecheckerColour1: '#000000',
    truecheckerColour2: '#ffffff',
    watermark: licenceConstraints.watermark ?? true,
    watermarkOpacity: (Array.isArray(licenceConstraints.watermark_opacity_range)
                           ? licenceConstraints.watermark_opacity_range[0]
                           : 1.0)
  };

  const keys   = Object.keys(defaults).filter(k => k !== 'watermark' && k !== 'watermarkOpacity');
  const stored = await chrome.storage.local.get(keys);
  const settings = { ...defaults, ...stored };

  // Populate UI
  effectSelect.value       = settings.blockoutEffect;
  blurRadiusInput.value    = settings.blurRadius;
  blurRadiusVal.textContent= settings.blurRadius;
  solidOpacityInput.value  = settings.solidOpacity;
  solidOpacityVal.textContent = settings.solidOpacity;
  solidColourInput.value   = settings.solidColour;
  pixelSizeSlider.value    = settings.pixelSize;
  pixelSizeVal.textContent = settings.pixelSize;
  watermarkEnabledInput.checked = settings.watermark;
  watermarkOpacityInput.value   = settings.watermarkOpacity;
  watermarkOpacityVal.textContent = settings.watermarkOpacity;

  // advanced UI:
  gridLineColourInput.value = settings.gridLineColour;
  gridBgColourInput.value   = settings.gridBgColour;
  gridBgOpacityInput.value  = settings.gridBgOpacity;
  gridBgOpacityVal.textContent = settings.gridBgOpacity;

  checkerColour1Input.value   = settings.checkerColour1;
  checkerColour2Input.value   = settings.checkerColour2;
  checkerSpacingSlider.value  = settings.checkerSpacing;
  checkerSpacingVal.textContent = settings.checkerSpacing;
  checkerOpacityInput.value   = settings.checkerOpacity;
  checkerOpacityVal.textContent = settings.checkerOpacity;

  truecheckerColour1Input.value = settings.truecheckerColour1;
  truecheckerColour2Input.value = settings.truecheckerColour2;

  moireStripeColour.value     = settings.moireStripeColour;
  moireBgOpacityInput.value   = settings.moireBgOpacity;
  moireBgOpacityVal.textContent  = settings.moireBgOpacity;

  dotsBgColourInput.value  = settings.dotsBgColour;
  dotsDotColourInput.value = settings.dotsDotColour;
  dotsSpacingSlider.value  = settings.dotsSpacing;
  dotsSpacingVal.textContent = settings.dotsSpacing;

  renderSliders(settings.blockoutEffect);
}

async function saveEffectSetting(key, value, displayUpdater) {
  await chrome.storage.local.set({ [key]: value });
  if (displayUpdater) displayUpdater(value);
  showStatus('Saved');
}

// ==== Profile Management ====
async function rebuildProfiles() {
  const { profiles = ['default'], activeProfile = 'default' } =
    await chrome.storage.local.get(['profiles', 'activeProfile']);

  profileSelect.innerHTML = '';
  profiles.forEach(p => {
    const opt = document.createElement('option');
    opt.value = p;
    opt.textContent = p;
    if (p === activeProfile) opt.selected = true;
    profileSelect.appendChild(opt);
  });
}

async function switchProfile(name) {
  await chrome.storage.local.set({ activeProfile: name });
  showStatus(`Switched to “${name}”`);
}

// ==== Import / Export ====
function exportProfiles() {
  chrome.storage.local.get(null, data => {
    const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
    const url  = URL.createObjectURL(blob);
    const a    = document.createElement('a');
    a.href     = url;
    a.download = 'srb_backup.json';
    a.click();
    URL.revokeObjectURL(url);
    showStatus('Exported');
  });
}

function importProfiles(file) {
  const reader = new FileReader();
  reader.onload = async () => {
    try {
      const obj = JSON.parse(reader.result);
      await chrome.storage.local.set(obj);
      showStatus('Imported');
      await initialize();  // reload UI state
    } catch {
      showStatus('Import failed', true);
    }
  };
  reader.readAsText(file);
}

// ==== Global Clear ====
async function clearCurrentProfile() {
  const { activeProfile = 'default' } = await chrome.storage.local.get('activeProfile');

  // Remove all region keys for this profile
  const all = await chrome.storage.local.get(null);
  const keys = Object.keys(all).filter(k =>
    k.startsWith(`srb_regions_${activeProfile}_`)
  );
  if (keys.length) await chrome.storage.local.remove(keys);

  showStatus('Cleared regions for current profile');
}

async function clearAllProfiles() {
  await chrome.storage.local.clear();
  showStatus('Cleared everything');
}

// ==== Event Bindings ==== =========================================
function bindListeners() {
  // Effect changes
  effectSelect.addEventListener('change', e => {
    saveEffectSetting('blockoutEffect', e.target.value, val => renderSliders(val));
  });

  blurRadiusInput.addEventListener('input', e => {
    blurRadiusVal.textContent = e.target.value;
  });
  blurRadiusInput.addEventListener('change', e => {
    saveEffectSetting('blurRadius', Number(e.target.value));
  });

  solidOpacityInput.addEventListener('input', e => {
    solidOpacityVal.textContent = e.target.value;
  });
  solidOpacityInput.addEventListener('change', e => {
    saveEffectSetting('solidOpacity', Number(e.target.value));
  });

  solidColourInput.addEventListener('change', e => {
    saveEffectSetting('solidColour', e.target.value);
  });

  pixelSizeSlider.addEventListener('input', e => {
    pixelSizeVal.textContent = e.target.value;
  });
  pixelSizeSlider.addEventListener('change', e => {
    saveEffectSetting('pixelSize', Number(e.target.value));
  });

// Grid
// Line Colour
gridLineColourInput.addEventListener('change', e => {
  chrome.storage.local.set({ gridLineColour: e.target.value });
  showStatus('Saved');
});

// Background Colour
gridBgColourInput.addEventListener('change', e => {
  chrome.storage.local.set({ gridBgColour: e.target.value });
  showStatus('Saved');
});

// Background Opacity
gridBgOpacityInput.addEventListener('input', e => {
  gridBgOpacityVal.textContent = e.target.value;
});
gridBgOpacityInput.addEventListener('change', e => {
  chrome.storage.local.set({ gridBgOpacity: Number(e.target.value) });
  showStatus('Saved');
});

// Spacing
checkerSpacingSlider.addEventListener('input', e => {
  checkerSpacingVal.textContent = e.target.value;
});
checkerSpacingSlider.addEventListener('change', e => {
  chrome.storage.local.set({ checkerSpacing: Number(e.target.value) });
  showStatus('Saved');
});

// Colour 1 & 2
checkerColour1Input.addEventListener('change', e => {
  chrome.storage.local.set({ checkerColour1: e.target.value });
  showStatus('Saved');
});
checkerColour2Input.addEventListener('change', e => {
  chrome.storage.local.set({ checkerColour2: e.target.value });
  showStatus('Saved');
});

// Opacity
checkerOpacityInput.addEventListener('input', e => {
  checkerOpacityVal.textContent = e.target.value;
});
checkerOpacityInput.addEventListener('change', e => {
  chrome.storage.local.set({ checkerOpacity: Number(e.target.value) });
  showStatus('Saved');
});

// Primary (first) colour
truecheckerColour1Input.addEventListener('change', e => {
  chrome.storage.local.set({ truecheckerColour1: e.target.value });
  showStatus('Saved');
});

// Secondary (second) colour
truecheckerColour2Input.addEventListener('change', e => {
  chrome.storage.local.set({ truecheckerColour2: e.target.value });
  showStatus('Saved');
});

// Moiré
moireStripeColour.addEventListener('change', e => {
  chrome.storage.local.set({ moireStripeColour: e.target.value });
  showStatus('Saved');
});
moireBgOpacityInput.addEventListener('input', e => {
  moireBgOpacityVal.textContent = e.target.value;
});
moireBgOpacityInput.addEventListener('change', e => {
  chrome.storage.local.set({ moireBgOpacity: Number(e.target.value) });
  showStatus('Saved');
});

// Dots
// Background colour
dotsBgColourInput.addEventListener('change', e => {
  chrome.storage.local.set({ dotsBgColour: e.target.value });
  showStatus('Saved');
});

// Dot colour
dotsDotColourInput.addEventListener('change', e => {
  chrome.storage.local.set({ dotsDotColour: e.target.value });
  showStatus('Saved');
});

// Spacing
dotsSpacingSlider.addEventListener('input', e => {
  dotsSpacingVal.textContent = e.target.value;
});
dotsSpacingSlider.addEventListener('change', e => {
  const v = Number(e.target.value);
  chrome.storage.local.set({ dotsSpacing: v });
  showStatus('Saved');
});

  // Import/Export
  exportBtn.addEventListener('click', exportProfiles);
  importBtn.addEventListener('click', () => importInput.click());
  importInput.addEventListener('change', e => {
    if (e.target.files[0]) importProfiles(e.target.files[0]);
  });

  // Global clear
  clearCurrentProfileBtn.addEventListener('click', () => {
    if (confirm('Clear regions on current profile?')) clearCurrentProfile();
  });
  clearAllProfilesBtn.addEventListener('click', () => {
    if (confirm('Clear all profiles & regions?')) clearAllProfiles();
  });

  // Profiles
  profileSelect.addEventListener('change', e => switchProfile(e.target.value));
  newProfileBtn.addEventListener('click', async () => {
    const name = prompt('Enter new profile name:');
    if (!name) return;
    const { profiles = [] } = await chrome.storage.local.get('profiles');
    if (profiles.includes(name)) {
      alert('Profile exists');
      return;
    }
    profiles.push(name);
    await chrome.storage.local.set({ profiles, activeProfile: name });
    await rebuildProfiles();
    showStatus(`Created “${name}”`);
  });
  deleteProfileBtn.addEventListener('click', async () => {
    const { profiles = [], activeProfile = 'default' } =
      await chrome.storage.local.get(['profiles', 'activeProfile']);
    if (activeProfile === 'default') {
      alert('Cannot delete default');
      return;
    }
    if (!confirm(`Delete profile “${activeProfile}”?`)) return;
    const newList = profiles.filter(p => p !== activeProfile);
    const newActive = newList[0] || 'default';
    await chrome.storage.local.set({ profiles: newList, activeProfile: newActive });
    await rebuildProfiles();
    showStatus(`Deleted`);
  });

// Toggle enable/disable watermark
watermarkEnabledInput.addEventListener('change', async e => {
  // 1) fetch the existing constraints
  const { licenceConstraints = {} } = await chrome.storage.local.get('licenceConstraints');
  // 2) update the watermark flag
  licenceConstraints.watermark = e.target.checked;
  // 3) write it back
  await chrome.storage.local.set({ licenceConstraints });
  showStatus('status_watermark_toggle_saved');
});

// Opacity slider
watermarkOpacityInput.addEventListener('change', async e => {
  const value = parseFloat(e.target.value);
  watermarkOpacityVal.textContent = value;
  // update the opacity range in the same licenceConstraints object
  const { licenceConstraints = {} } = await chrome.storage.local.get('licenceConstraints');
  licenceConstraints.watermark_opacity_range = [value];
  await chrome.storage.local.set({ licenceConstraints });
  showStatus('status_watermark_saved');
});

const customInput  = document.getElementById('customWatermarkInput');
const previewDiv   = document.getElementById('customWatermarkPreview');

customInput.addEventListener('change', async e => {
  const file = e.target.files[0];
  if (!file) return;

  // 1) validate type
  if (!['image/png','image/jpeg'].includes(file.type)) {
    alert('Only PNG or JPEG allowed.');
    customInput.value = '';
    return;
  }
  // 2) validate size ≤200 KB
  if (file.size > 200 * 1024) {
    alert('File must be ≤200 KB.');
    customInput.value = '';
    return;
  }

  // 3) load into an Image to enforce max dimensions 256×256
  const dataUrl = await new Promise((res, rej) => {
    const reader = new FileReader();
    reader.onload = () => res(reader.result);
    reader.onerror = () => rej(reader.error);
    reader.readAsDataURL(file);
  });
  const img = new Image();
  img.onload = async () => {
    if (img.width > 256 || img.height > 256) {
      alert('Image must be no larger than 256×256px.');
      customInput.value = '';
      return;
    }
    // 4) save to storage
    await chrome.storage.local.set({ customWatermark: dataUrl });
    showStatus('Custom watermark saved');
    // 5) preview it
    previewDiv.innerHTML = '';
    previewDiv.appendChild(img);
  };
  img.src = dataUrl;
});

}

//—
// 4) Tier UI initialiser
// ——————————————————————————————————————————————
/**
 * Enable/disable or show/hide bits of the options page based on
 * the licenceType and its constraints.
 *
 * @param {string} type  "Lite" | "Standard" | "Anonymous"
 * @param {object} c            The constraint set for that type
 */
async function initialiseOptionsForTier(licenceType, constraints) {


switch (licenceType) {

  case 'Lite':
  //adjust profile dropdown
    optImpExp && (optImpExp.disabled = !constraints.allowed_profiles);
    exportBtn && (exportBtn.disabled = !constraints.allowed_profiles);
    importBtn && (importBtn.disabled = !constraints.allowed_profiles);

    optGlobalClear && (optGlobalClear.disabled = !constraints.allowed_profiles);
    clearCurrentProfileBtn && (clearCurrentProfileBtn.disabled = !constraints.allowed_profiles);
    clearAllProfilesBtn && (clearAllProfilesBtn.disabled = !constraints.allowed_profiles);

    optProfiles && (optProfiles.disabled = !constraints.allowed_profiles);
    profileSelect && (profileSelect.disabled = !constraints.allowed_profiles);
    newProfileBtn && (newProfileBtn.disabled = !constraints.allowed_profiles);
    deleteProfileBtn && (deleteProfileBtn.disabled = !constraints.allowed_profiles);

    watermarkSection && (watermarkSection.disabled = constraints.watermark);
    watermarkOpacityVal && (watermarkOpacityVal.value = constraints.watermark_opacity_range);
    watermarkOpacityInput && (watermarkOpacityInput.disabled = constraints.watermark);
    watermarkEnabledInput && (watermarkEnabled.checked = constraints.watermark);
    watermarkEnabledInput && (watermarkEnabledInput.disabled = constraints.watermark);

    blurRadiusInput && (blurRadius.disabled = !constraints.blur_slider);

  document.getElementById('customWatermarkSection')
    .classList.add('hidden');

  break;


  case 'Standard':
  //adjust profile dropdown
    optImpExp && (optImpExp.disabled = !constraints.allowed_profiles);
    exportBtn && (exportBtn.disabled = !constraints.allowed_profiles);
    importBtn && (importBtn.disabled = !constraints.allowed_profiles);

    optGlobalClear && (optGlobalClear.disabled = !constraints.allowed_profiles);
    clearCurrentProfileBtn && (clearCurrentProfileBtn.disabled = !constraints.allowed_profiles);
    clearAllProfilesBtn && (clearAllProfilesBtn.disabled = !constraints.allowed_profiles);

    optProfiles && (optProfiles.disabled = !constraints.allowed_profiles);
    profileSelect && (profileSelect.disabled = !constraints.allowed_profiles);
    newProfileBtn && (newProfileBtn.disabled = !constraints.allowed_profiles);
    deleteProfileBtn && (deleteProfileBtn.disabled = !constraints.allowed_profiles);

  // 1) Disable the entire watermark section
  if (watermarkSection) {
    // give it the not‑allowed cursor
    watermarkSection.style.cursor = 'not-allowed';
    // visually dim it
    watermarkSection.style.opacity = '0.5';
    // insert a notice at the top
    watermarkSection.innerHTML = `
      <p style="margin:0 0 .5em;font-size:14px;font-style:italic;color:#0f0f0f">
        Watermark settings are not alterable on the Standard licence.
      </p>
      ${watermarkSection.innerHTML}
    `;
    // disable its form controls
    watermarkEnabled.checked = true;
    watermarkEnabledInput.disabled = true;
    watermarkOpacityInput.disabled = true;
  }

    blurRadiusInput && (blurRadius.disabled = !constraints.blur_slider);

  document.getElementById('customWatermarkSection')
    .classList.add('hidden');

  break;

// ----------------------------------------------------------------------------------
  case 'Anonymous':
  //adjust profile dropdown
    optImpExp && (optImpExp.disabled = !constraints.allowed_profiles);
    exportBtn && (exportBtn.disabled = !constraints.allowed_profiles);
    importBtn && (importBtn.disabled = !constraints.allowed_profiles);

    optGlobalClear && (optGlobalClear.disabled = !constraints.allowed_profiles);
    clearCurrentProfileBtn && (clearCurrentProfileBtn.disabled = !constraints.allowed_profiles);
    clearAllProfilesBtn && (clearAllProfilesBtn.disabled = !constraints.allowed_profiles);

    optProfiles && (optProfiles.disabled = !constraints.allowed_profiles);
    profileSelect && (profileSelect.disabled = !constraints.allowed_profiles);
    newProfileBtn && (newProfileBtn.disabled = !constraints.allowed_profiles);
    deleteProfileBtn && (deleteProfileBtn.disabled = !constraints.allowed_profiles);

    watermarkSection && (watermarkSection.disabled = !constraints.watermark);
    watermarkOpacityVal && (watermarkOpacityVal.value = !constraints.watermark_opacity_range);
    watermarkOpacityInput && (watermarkOpacityInput.disabled = !constraints.watermark);
    watermarkEnabledInput && (watermarkEnabled.checked = !constraints.watermark);
    watermarkEnabledInput && (watermarkEnabledInput.disabled = !constraints.watermark);

    blurRadiusInput && (blurRadius.disabled = !constraints.blur_slider);

  document.getElementById('customWatermarkSection')
    .classList.remove('hidden');

  break;

}

}

// ==== Initialise UI =======================================================
async function initialise(settings) {
  const {
    activeProfile = 'default',
    browserId = 'none',
    browserInfo = 'none',
    deviceFingerprint = 'none',
    deviceId = 'none',
    fingerprint = 'none',
    licenceConstraints = {},
    licenceLastChecked = 'none',
    licenceType = {},
    os = 'none',
    profiles = {},
    timezoneLocale = 'none'
  } = settings;


  // 1) initialise tier UI
  const c = TIER_CONSTRAINTS[licenceType] || TIER_CONSTRAINTS.Lite;
  await initialiseOptionsForTier(licenceType, c);


async function populateEffectSelect() {
  const { licenceConstraints } = await chrome.storage.local.get('licenceConstraints');

  const allowedEffects = licenceConstraints?.allowed_effects || ['blur']; // fallback to 'blur'

  const effectSelect = document.getElementById('effectSelect');
  if (!effectSelect) {
    console.warn('[SRB] effectSelect element not found');
    return;
  }

  // Clear existing options
  effectSelect.innerHTML = '';

  // Populate allowed effects
  allowedEffects.forEach(effect => {
    const option = document.createElement('option');
    option.value = effect;
    option.textContent = effect.charAt(0).toUpperCase() + effect.slice(1);
    effectSelect.appendChild(option);
  });

  // Restore last saved selection if still allowed
  const { selectedEffect } = await chrome.storage.local.get('selectedEffect');
  if (selectedEffect && allowedEffects.includes(selectedEffect)) {
    effectSelect.value = selectedEffect;
  }
}

  // Banner when in Lite mode:
  const banner = document.getElementById('liteModeBanner');
  if (licenceType === 'Lite' && banner) {
    banner.classList.remove('hidden');
  } else {
    banner && banner.classList.add('hidden');
  }

  // Ensure profiles exist
  const data = await chrome.storage.local.get(['profiles','activeProfile']);
  if (!Array.isArray(data.profiles)) {
    await chrome.storage.local.set({ profiles: ['default'], activeProfile: 'default' });
  }

  populateEffectSelect();
  await rebuildProfiles();
  await loadEffectSettings();
  bindListeners();
}


// Let's get started..
document.addEventListener('DOMContentLoaded', initialise);
