// popup.js
import { DEFAULT_EFFECT_SETTINGS, DEFAULT_CONSTRAINTS } from './licencing/constants.js';
import { checkAndCacheLicence } from './licencing/licence-check.js';

const API_BASE = 'https://www.jaysndees.com/extensions/srb_v2043/api';

const LITE_CONSTRAINTS = {
    max_regions_per_page: 3,
    max_pages: 2,
    allowed_profiles: false,
    max_profiles: 1,
    allowed_effects: ["blur"],
    blur_slider: false,
    import_export: false,
    activation_limit: 1,
    screenshot: false,
    watermark: true,
    watermark_slider: false,
    watermark_toggle: false,
    watermark_opacity_range: [1.0],
    color_picker: false,
    pixel_able: false,
    pixel_size: 0
  };

// === Utilities ===

function showToast(message) {
  const toast = document.getElementById('toast');
  if (!toast) return;
  toast.textContent = message;
  toast.classList.remove('hidden');
  setTimeout(() => toast.classList.add('hidden'), 3000);
}

function openExtensionPage(file) {
  const url = chrome.runtime.getURL(file);
  window.open(url, '_blank');
}

async function callApi(path, data) {
  const res = await fetch(`${API_BASE}/${path}`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
    body: new URLSearchParams(data)
  });
  const json = await res.json();
  return { status: res.status, ...json };
}

function getEnvInfo() {
  return {
    browserId: chrome.runtime.id,
    deviceFingerprint: navigator.userAgent,
    extensionVersion: chrome.runtime.getManifest().version,
    os: navigator.platform
  };
}

async function getActiveTabId() {
  const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
  return tabs[0]?.id;
}

function sendMessageToTab(tabId, message) {
  return new Promise((resolve, reject) => {
    chrome.tabs.sendMessage(tabId, message, response => {
      if (chrome.runtime.lastError) return reject(chrome.runtime.lastError);
      resolve(response);
    });
  });
}


// ----------- UPDATED SCREENSHOT
async function captureScreenshot() {
  try {
    const screenshotDataUrl = await chrome.tabs.captureVisibleTab(null, { format: 'png' });
    console.log('[Screenshot] Captured screenshot:', screenshotDataUrl);
    return screenshotDataUrl;
  } catch (err) {
    console.error('[Screenshot] Failed to capture screenshot:', err);
    throw err;
  }
}

async function handleScreenshotClick() {
  const storage = await chrome.storage.local.get(['licenceConstraints', 'watermarkToggle', 'watermarkOpacity']);
  const constraints = storage.licenceConstraints || {};

  if (!constraints.screenshot) {
    alert('Your current licence does not allow screenshot functionality.');
    return;
  }

  try {
    const screenshotDataUrl = await captureScreenshot();

    // Determine if watermark should be applied
    const shouldWatermark = constraints.watermark || storage.watermarkToggle === true;
    const opacity = typeof storage.watermarkOpacity === 'number' ? storage.watermarkOpacity : 1.0;

    const finalImage = shouldWatermark? await applyWatermarkToImage(screenshotDataUrl, opacity)
      : screenshotDataUrl;

    // Download it
    const blob = await (await fetch(finalImage)).blob();
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `sRB_screenshot_${Date.now()}.png`;
    a.click();
    URL.revokeObjectURL(url);
  } catch (err) {
    console.error('[Screenshot] Error:', err);
    alert('Screenshot failed.');
  }
}

async function applyWatermarkToImage(imageDataUrl, opacity = 1.0) {
  // 1) figure out which watermark src to use
  const { licenceType, customWatermark } =
    await chrome.storage.local.get(['licenceType','customWatermark']);

  // only Anonymous may have a custom watermark
  const watermarkSrc =
    licenceType === 'Anonymous' && customWatermark
      ? customWatermark
      : chrome.runtime.getURL('icons/favicon-128x128.png');

  // 2) load the screenshot
  const screenshotImg = await new Promise((res, rej) => {
    const img = new Image();
    img.onload  = () => res(img);
    img.onerror = () => rej(new Error('Failed to load screenshot'));
    img.src     = imageDataUrl;
  });

  // 3) load the watermark
  const watermarkImg = await new Promise((res, rej) => {
    const img = new Image();
    img.onload  = () => res(img);
    img.onerror = () => rej(new Error('Failed to load watermark'));
    img.src     = watermarkSrc;
  });

  // 4) composite onto a same‑sized canvas
  const canvas = document.createElement('canvas');
  canvas.width  = screenshotImg.width;
  canvas.height = screenshotImg.height;
  const ctx = canvas.getContext('2d');

  // draw the screenshot
  ctx.drawImage(screenshotImg, 0, 0);

  // compute watermark size & position
  const margin = 16;
  const scale  = 0.25;
  const wW     = screenshotImg.width * scale;
  const wH     = (watermarkImg.height / watermarkImg.width) * wW;
  const x      = screenshotImg.width - wW - margin;
  const y      = screenshotImg.height - wH - margin;

  // draw it with the requested opacity
  ctx.globalAlpha = opacity;
  ctx.drawImage(watermarkImg, x, y, wW, wH);
  ctx.globalAlpha = 1;

  // 5) return the combined PNG
  return canvas.toDataURL('image/png');
}

// === DOM Ready ===
document.addEventListener('DOMContentLoaded', async () => {

const {
        licenceType = 'Lite',
        licenceConstraints: constraints = {}
    } = await chrome.storage.local.get(['licenceType', 'licenceConstraints']);

  const addRegionBtn = document.getElementById('addRegion');
  const clearRegionsBtn = document.getElementById('clearRegions');
  const clearAllRegionsBtn = document.getElementById('clearAllRegions');
  const refreshRegionsBtn = document.getElementById('refreshRegions');
  const screenshotBtn = document.getElementById('processScreenshot');
  const profileSelect = document.getElementById('profileSelect');
  const newProfileBtn = document.getElementById('newProfile');
  const deleteProfileBtn = document.getElementById('deleteProfile');
  const openOptionsBtn = document.getElementById('openOptions');
  const helpBtn = document.getElementById('openHelpMain');
  const aboutBtn    = document.getElementById('openAbout');
  const aboutModal  = document.getElementById('aboutModal');
  const closeAbout  = document.getElementById('closeAbout');
  const licenceDisplay = document.getElementById('licenceType');
  const versionDisplay = document.getElementById('versionValid');
  const registerBtn = document.getElementById('registerManageLicence');
  const deactivateBtn = document.getElementById('deactivateSelf');
  const liteBanner = document.getElementById('liteModeBanner');

  const aboutVersionEl = document.getElementById('aboutVersion');
  if (aboutVersionEl) {
    aboutVersionEl.textContent = chrome.runtime.getManifest().version;
  }

    //const { licenceType = 'Lite' } = await chrome.storage.local.get('licenceType');

    if (licenceType === 'Lite') {
        liteBanner.classList.remove('hidden');
    } else {
        liteBanner.classList.add('hidden');
        deactivateBtn.classList.remove('hidden');
	if (licenceType === 'Anonymous'){
		registerBtn.classList.add('hidden');
	}
    }

  licenceDisplay.textContent = licenceType;
  versionDisplay.textContent = "version: " + chrome.runtime.getManifest().version;

    // === Enforce constraints ===
    if (Object.keys(constraints).length > 0) {
        if (!constraints.screenshot && screenshotBtn) {
            screenshotBtn.disabled = true;
            screenshotBtn.title = 'Screenshots are not available on your licence.';
        }
        if (!constraints.allowed_profiles) {
            newProfileBtn?.setAttribute('disabled', 'disabled');
            newProfileBtn.title = 'Profile Management is not available on your licence.';
            deleteProfileBtn?.setAttribute('disabled', 'disabled');
            deleteProfileBtn.title = 'Profile Management is not available on your licence.';
            profileSelect?.setAttribute('disabled', 'disabled');
            profileSelect.title = 'Profile Management is not available on your licence.';
        }

    }

    // === Populate profiles ===
    async function populateProfiles() {
        const { profiles = ['default'], activeProfile = 'default' } = await chrome.storage.local.get(['profiles', 'activeProfile']);
        profileSelect.innerHTML = ''; // clear

        profiles.forEach(profile => {
            const opt = document.createElement('option');
            opt.value = profile;
            opt.textContent = profile;
            if (profile === activeProfile) {
                opt.selected = true;
            }
            profileSelect.appendChild(opt);
        });
    }

    await populateProfiles();

    // === Other initialisation and event listeners can continue here ===


    // Button: Register / Manage Licence
    registerBtn.addEventListener('click', () => {
        chrome.tabs.create({
            url: chrome.runtime.getURL('register.html')
        });
    });

  helpBtn?.addEventListener('click', () => openExtensionPage('help.html'));

  openOptionsBtn?.addEventListener('click', () => {
    chrome.runtime.openOptionsPage?.() ?? openExtensionPage('options.html');
  });

  addRegionBtn?.addEventListener('click', async () => {
    const tabId = await getActiveTabId();
    if (!tabId) return showToast('No active tab found');

    try {
      const { constraints } = await checkAndCacheLicence();
      const resp = await sendMessageToTab(tabId, { action: 'getRegionCount' });
      const count = resp?.count ?? 0;

      if (constraints.max_regions_per_page >= 0 && count >= constraints.max_regions_per_page) {
        return showToast(`Max ${constraints.max_regions_per_page} regions allowed`);
      }

      await sendMessageToTab(tabId, { action: 'addRegion' });
      showToast('Armed for region draw...');
      setTimeout(() => window.close(), 1000);
    } catch (e) {
      console.error(e);
      showToast('Error adding region');
    }
  });

  clearRegionsBtn?.addEventListener('click', async () => {
    if (!confirm('Clear all regions on this page?')) return;
    const tabId = await getActiveTabId();
    if (!tabId) return showToast('No active tab found');
    await sendMessageToTab(tabId, { action: 'clearRegions' });
    showToast('Regions cleared on this page');
  });

  clearAllRegionsBtn?.addEventListener('click', async () => {
    if (!confirm('Remove all regions for this profile?')) return;
    const tabId = await getActiveTabId();
    if (!tabId) return showToast('No active tab found');
    await sendMessageToTab(tabId, { action: 'clearAllRegions' });
    showToast('All regions cleared for this profile');
  });

  // — “Refresh Regions” button
refreshRegionsBtn?.addEventListener('click', () => {
  chrome.tabs.query({ active: true, currentWindow: true }, ([tab]) => {
    if (!tab?.id) return showToast('No active tab found');
    
    chrome.tabs.sendMessage(tab.id, { action: 'refreshRegions' }, response => {
      if (chrome.runtime.lastError) {
        // no listener in content.js
        return showToast('This page isn’t supported');
      }
      showToast('Regions refreshed');
    });
  });
})

aboutBtn?.addEventListener('click', () => {
  aboutModal.classList.remove('hidden');
});

closeAbout?.addEventListener('click', () => {
  aboutModal.classList.add('hidden');
});

// also close if the user clicks outside the modal box
aboutModal?.addEventListener('click', e => {
  if (e.target === aboutModal) {
    aboutModal.classList.add('hidden');
  }
});


// — “Screenshot” button
document.getElementById('processScreenshot').addEventListener('click', handleScreenshotClick);

  profileSelect?.addEventListener('change', async e => {
    const profile = e.target.value;
    await chrome.storage.local.set({ activeProfile: profile });
    showToast(`Switched to profile: ${profile}`);
    const tabId = await getActiveTabId();
    if (tabId) await sendMessageToTab(tabId, { action: 'refreshRegions' });
  });

  newProfileBtn?.addEventListener('click', async () => {
    const name = prompt('Enter a name for the new profile:');
    if (!name) return;
    const { profiles = ['default'] } = await chrome.storage.local.get('profiles');
    if (!profiles.includes('default')) {
      profiles.unshift('default');
    }
    if (profiles.includes(name)) {
      alert('Profile already exists.');
      return;
    }
    profiles.push(name);
    await chrome.storage.local.set({ profiles, activeProfile: name });
    await populateProfiles();
    showToast(`Profile "${name}" created`);
    // (optionally, trigger a page‐refresh of regions)
    const tabId = await getActiveTabId();
    if (tabId) await sendMessageToTab(tabId, { action: 'refreshRegions' });

  });

  deactivateBtn?.addEventListener('click', async () => {
    if (!confirm('Deactivate your licence on ALL devices?')) return;

    // 1️⃣ Get the stored licence key
    const { licenceKey } = await chrome.storage.local.get('licenceKey');
    if (!licenceKey) {
      showToast('No licence is currently active.');
      return;
    }
    // 1️⃣.1️⃣ Get the stored browserId
    const { browserId } = await chrome.storage.local.get('browserId');
    if (!browserId) {
      showToast('No browserId found!.');
      return;
    }
    try {
      // 2️⃣ Call the deactivation endpoint
      const resp = await fetch(`${API_BASE}/deactivate_licence.php`, {
        method:  'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body:    new URLSearchParams({ licenceKey, browserId })
      });
      const data = await resp.json();

      if (resp.ok && data.success) {
        // 3️⃣ On success, clear licence data and reset to Lite
        await chrome.storage.local.remove('licenceKey');
        await chrome.storage.local.set({
          licenceType:        'Lite',
          licenceConstraints: LITE_CONSTRAINTS,
          activeProfile:      'default',
          profiles:           ['default']
        });

        // Remove any non‑default region keys
        const allKeys = await chrome.storage.local.get(null);
        const toRemove = Object.keys(allKeys)
          .filter(k => k.startsWith('srb_regions_') && !k.includes('_default_'));
        if (toRemove.length) {
          await chrome.storage.local.remove(toRemove);
        }

// ------        // Remove any non‑Lite region keys
        const allAdv = await chrome.storage.local.get(null);
        const prefixes = [
  		'checker', 
  		'grid',
  		'moire',
  		'dots',
  		'truechecker'
	];
        const advRemove = Object.keys(allAdv)
          .filter(key => prefixes.some(prefix => key.startsWith(prefix)));
        if (advRemove.length) {
          await chrome.storage.local.remove(advRemove);
        }

	// 5) Reset every default‑profile region back to pure blur
        const afterAdv = await chrome.storage.local.get(null);
	const defaultRegionKeys = Object.keys(afterAdv)
  	.filter(k => k.startsWith('srb_regions_default_'));

	for (const key of defaultRegionKeys) {
  		const { [key]: regions = [] } = await chrome.storage.local.get(key);
  		const resetRegions = regions.map(r => ({
    		top:    r.top,
    		left:   r.left,
    		width:  r.width,
    		height: r.height,
    		// blow away any old effect and use the blur defaults
    		effect: { ...DEFAULT_EFFECT_SETTINGS }
  		}));
 		 if (resetRegions.length) {
    		await chrome.storage.local.set({ [key]: resetRegions });
  		}
	}


	// 🔄 TRIM EXCESS REGIONS FOR THE DEFAULT PROFILE
	const max = DEFAULT_CONSTRAINTS.max_regions_per_page;  // e.g. 3
	// 1️⃣ Fetch every key
	const all = await chrome.storage.local.get(null);
	// 2️⃣ Find only the default profile page‐keys
	const defaultPrefix = 'srb_regions_default_';
	const keysToTrim   = Object.keys(all).filter(k => k.startsWith(defaultPrefix));

	for (const key of keysToTrim) {
	  const regions = all[key] || [];
	  if (regions.length > max) {
	    // 3️⃣ Truncate to the allowed maximum
	    const trimmed = regions.slice(0, max);
	    // 4️⃣ Write it back
 	   await chrome.storage.local.set({ [key]: trimmed });
	  }
	}

  	liteBanner.classList.remove('hidden');
  	licenceDisplay.textContent = 'Lite';
  	versionDisplay.textContent = "version: " + chrome.runtime.getManifest().version;
	deactivateBtn.classList.add('hidden');
        showToast('Licence deactivated. Running in Lite mode.');

      } else {
        // Server‑side failure
        showToast('Deactivation failed: ' + (data.error || 'Unknown'));
      }
    } catch (err) {
      console.error('Deactivation error', err);
      showToast('Network error while deactivating.');
    }
  });

  deleteProfileBtn?.addEventListener('click', async () => {
    const { profiles = [], activeProfile = 'default' } =
      await chrome.storage.local.get(['profiles', 'activeProfile']);
    if (activeProfile === 'default') {
      alert('Cannot delete the default profile.');
      return;
    }
    if (!confirm(`Delete profile "${activeProfile}"?`)) return;
    const deletedProfile = activeProfile;
    const updated = profiles.filter(p => p !== deletedProfile);
    const nextActive = updated[0] || 'default';
    await chrome.storage.local.set({ profiles: updated, activeProfile: nextActive });
    const allKeys = await chrome.storage.local.get(null);
    const regionKeysToRemove = Object.keys(allKeys).filter(key =>
      key.startsWith(`srb_regions_${deletedProfile}_`)
    );
    if (regionKeysToRemove.length) {
      await chrome.storage.local.remove(regionKeysToRemove);
    }
    showToast(`Profile "${deletedProfile}" deleted successfully and its regions removed`);
    // **→ refresh the dropdown to pick up the change**
    await populateProfiles();

    // (optionally, trigger a page‐refresh of regions)
    const tabId = await getActiveTabId();
    if (tabId) await sendMessageToTab(tabId, { action: 'refreshRegions' });
  });
});